<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

header('Content-Type: application/json');

if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Not authorized']);
    exit;
}

try {
    if (!isset($_GET['sale_id'])) {
        throw new Exception('Sale ID is required');
    }

    $sale_id = $_GET['sale_id'];

    // Get sale details and check if it's a credit sale
    $stmt = $pdo->prepare("
        SELECT 
            s.id,
            s.customer_id,
            s.payment_method,
            s.total_amount,
            c.name as customer_name,
            ct.id as credit_transaction_id,
            ct.amount as credit_amount
        FROM sales s
        LEFT JOIN customers c ON s.customer_id = c.id
        LEFT JOIN credit_transactions ct ON s.id = ct.sale_id
        WHERE s.id = ?
    ");
    $stmt->execute([$sale_id]);
    $sale = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$sale) {
        throw new Exception('Sale not found');
    }

    // Get multiple payment methods from sale_payments table
    $stmt = $pdo->prepare("SELECT method, amount FROM sale_payments WHERE sale_id = ? ORDER BY created_at ASC");
    $stmt->execute([$sale_id]);
    $sale_payments = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Calculate payment breakdown (credit is debt, not payment)
    $payment_breakdown = [];
    $total_paid = 0;
    $credit_amount = 0;
    
    foreach ($sale_payments as $payment) {
        $method = strtolower($payment['method']);
        $amount = floatval($payment['amount']);
        
        if ($method === 'credit') {
            // Credit is debt, not payment
            $credit_amount = $amount;
            $payment_breakdown[] = [
                'method' => 'Credit',
                'amount' => $amount,
                'is_debt' => true
            ];
        } else {
            // Real payments (cash, mpesa, etc.)
            $payment_breakdown[] = [
                'method' => ucfirst($method),
                'amount' => $amount,
                'is_debt' => false
            ];
            $total_paid += $amount;
        }
    }
    
    // If no payment methods found, use the primary payment method
    if (empty($payment_breakdown)) {
        $is_credit = strtolower($sale['payment_method']) === 'credit';
        $payment_breakdown[] = [
            'method' => ucfirst($sale['payment_method']),
            'amount' => floatval($sale['total_amount']),
            'is_debt' => $is_credit
        ];
        if (!$is_credit) {
            $total_paid = floatval($sale['total_amount']);
        } else {
            $credit_amount = floatval($sale['total_amount']);
        }
    }
    
    $response = [
        'success' => true,
        'sale_id' => $sale['id'],
        'customer_id' => $sale['customer_id'],
        'customer_name' => $sale['customer_name'],
        'payment_method' => $sale['payment_method'],
        'total_amount' => floatval($sale['total_amount']),
        'payment_methods' => $payment_breakdown,
        'total_paid' => $total_paid,
        'credit_amount' => $credit_amount,
        'has_multiple_payments' => count($payment_breakdown) > 1,
        'is_credit_sale' => false,
        'credit_balance' => 0,
        'credit_transaction_id' => null
    ];

    // If it's a credit sale, calculate the remaining balance
    if (($sale['payment_method'] === 'credit' || $credit_amount > 0) && $sale['credit_transaction_id']) {
        $stmt = $pdo->prepare("
            SELECT 
                ct.amount,
                COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as paid_amount,
                COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived_amount
            FROM credit_transactions ct
            LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
            WHERE ct.id = ?
            GROUP BY ct.id, ct.amount
        ");
        $stmt->execute([$sale['credit_transaction_id']]);
        $credit_info = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($credit_info) {
            $credit_balance = $credit_info['amount'] - $credit_info['paid_amount'] - $credit_info['waived_amount'];
            
            $response['is_credit_sale'] = true;
            $response['credit_balance'] = max(0, $credit_balance);
            $response['credit_transaction_id'] = $sale['credit_transaction_id'];
            $response['original_credit_amount'] = floatval($credit_info['amount']);
            $response['paid_amount'] = floatval($credit_info['paid_amount']);
            $response['waived_amount'] = floatval($credit_info['waived_amount']);
        }
    }

    echo json_encode($response);

} catch (Exception $e) {
    error_log("Check Sale Credit Balance Error: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'Error checking credit balance: ' . $e->getMessage()
    ]);
} 